/**
 * SERPforge SEO - Block Editor Integration
 * (Vanilla JS / createElement version)
 */
(function (wp) {
    if (!wp || !wp.plugins) {
        console.error('SERPforge SEO: WP Plugins API missing');
        return;
    }

    const { registerPlugin } = wp.plugins;
    const { PluginSidebar, PluginSidebarMoreMenuItem } = wp.editor;
    const { PanelBody, TextControl, TextareaControl, SelectControl, CheckboxControl, TabPanel, Button } = wp.components;
    const { useSelect, useDispatch } = wp.data;
    const { useState, useEffect, createElement: el, Fragment } = wp.element;
    const { __ } = wp.i18n;

    const SERPFORGE_ICON = 'chart-line';

    // --- STYLES & HELPERS ---
    const getStatusColor = (status) => {
        switch (status) {
            case 'good': return '#46b450';
            case 'improve': return '#ffb900';
            case 'critical': return '#dc3232';
            default: return '#999';
        }
    };

    // --- COMPONENTS ---
    // Defined outside the main component to prevent re-mounting (loss of focus) on re-renders.

    const ScoreCard = ({ score }) => {
        return el('div', {
            style: {
                background: score >= 80 ? 'linear-gradient(135deg, #46b450 0%, #2ea33a 100%)' :
                    (score >= 50 ? 'linear-gradient(135deg, #ffb900 0%, #f5a623 100%)' :
                        'linear-gradient(135deg, #dc3232 0%, #b32d2e 100%)'),
                borderRadius: '8px',
                padding: '20px',
                textAlign: 'center',
                marginBottom: '20px',
                color: '#fff',
                boxShadow: '0 4px 6px rgba(0,0,0,0.1)'
            }
        },
            el('div', { style: { fontSize: '42px', fontWeight: '800', lineHeight: '1' } }, `${score}`),
            el('div', { style: { fontSize: '12px', opacity: '0.9', textTransform: 'uppercase', letterSpacing: '1px', marginTop: '5px' } }, 'SEO Score')
        );
    };

    const SnippetPreview = ({ meta, title, permalink }) => {
        const displayTitle = meta['_serpforge_seo_title'] || title || 'Post Title';
        // Simple variable replacement
        const cleanTitle = displayTitle.replace('%title%', title || '').replace('%sep%', '-').replace('%sitename%', 'Site Name');

        const displayDesc = meta['_serpforge_seo_description'] || 'Please provide a meta description.';
        const displayUrl = permalink || 'http://example.com/post-slug';

        return el('div', {
            className: 'serpforge-snippet-preview', style: {
                background: '#fff', border: '1px solid #dfe1e5', borderRadius: '8px', padding: '15px', marginBottom: '20px', fontFamily: 'arial,sans-serif'
            }
        },
            el('div', { style: { fontSize: '14px', color: '#202124', marginBottom: '4px' } }, 'Google Preview'),
            el('div', { style: { fontSize: '12px', color: '#5f6368', whiteSpace: 'nowrap', overflow: 'hidden', textOverflow: 'ellipsis' } }, displayUrl),
            el('div', { style: { fontSize: '20px', color: '#1a0dab', lineHeight: '1.3', cursor: 'pointer', marginBottom: '3px' } }, cleanTitle),
            el('div', { style: { fontSize: '14px', color: '#545454', lineHeight: '1.4' } }, displayDesc)
        );
    };

    const GeneralTab = ({ meta, updateMeta, title, permalink, checks }) => {
        return el('div', { style: { marginTop: '15px' } },
            el(PanelBody, { title: 'Basic Settings', initialOpen: true },
                el(SnippetPreview, { meta, title, permalink }),
                el(TextControl, {
                    label: 'Focus Keyword',
                    value: meta['_serpforge_focus_keyword'] || '',
                    onChange: (val) => updateMeta('_serpforge_focus_keyword', val),
                    help: 'Main keyword for optimization.',
                    __next40pxDefaultSize: true,
                    __nextHasNoMarginBottom: true
                }),
                el('hr', { className: 'serpforge-divider' }),
                el(TextControl, {
                    label: 'SEO Title',
                    value: meta['_serpforge_seo_title'] || '',
                    placeholder: title,
                    onChange: (val) => updateMeta('_serpforge_seo_title', val),
                    __next40pxDefaultSize: true,
                    __nextHasNoMarginBottom: true
                }),
                el(TextareaControl, {
                    label: 'Meta Description',
                    value: meta['_serpforge_seo_description'] || '',
                    onChange: (val) => updateMeta('_serpforge_seo_description', val),
                    rows: 3,
                    __next40pxDefaultSize: true,
                    __nextHasNoMarginBottom: true
                })
            ),
            el(PanelBody, { title: 'Analysis', initialOpen: true },
                el('div', { className: 'serpforge-analysis-list' },
                    checks.length > 0 ? el('ul', { style: { margin: 0, padding: 0, listStyle: 'none' } },
                        checks.map(check => el('li', {
                            key: check.id,
                            style: {
                                display: 'flex',
                                alignItems: 'center',
                                marginBottom: '10px',
                                padding: '8px',
                                background: '#f8f9fa',
                                borderRadius: '4px',
                                borderLeft: `4px solid ${getStatusColor(check.status)}`
                            }
                        },
                            el('span', { style: { fontSize: '13px', fontWeight: '500' } }, check.label)
                        ))
                    ) : el('p', { style: { fontStyle: 'italic', color: '#666', textAlign: 'center' } }, 'Add content to start analysis.')
                )
            )
        );
    };

    const AdvancedTab = ({ meta, updateMeta }) => {
        const robots = meta['_serpforge_robots'] || [];

        const updateRobots = (value, checked) => {
            let newRobots = [...robots];
            if (checked) {
                if (!newRobots.includes(value)) newRobots.push(value);
            } else {
                newRobots = newRobots.filter(r => r !== value);
            }
            updateMeta('_serpforge_robots', newRobots);
        };

        return el('div', { style: { marginTop: '15px' } },
            el(PanelBody, { title: 'Robots Meta', initialOpen: true },
                el(CheckboxControl, {
                    label: 'Disable Image SEO',
                    checked: meta['_serpforge_disable_image_seo'] === '1',
                    onChange: (isChecked) => updateMeta('_serpforge_disable_image_seo', isChecked ? '1' : ''),
                    help: 'Do not auto-add ALT/Title attributes for this post.'
                }),
                el(CheckboxControl, {
                    label: 'No Index (noindex)',
                    checked: robots.includes('noindex'),
                    onChange: (isChecked) => updateRobots('noindex', isChecked),
                    help: 'Tell search engines not to index this page.'
                }),
                el(CheckboxControl, {
                    label: 'No Follow (nofollow)',
                    checked: robots.includes('nofollow'),
                    onChange: (isChecked) => updateRobots('nofollow', isChecked),
                    help: 'Tell search engines not to follow links on this page.'
                }),
                el(CheckboxControl, {
                    label: 'No Archive (noarchive)',
                    checked: robots.includes('noarchive'),
                    onChange: (isChecked) => updateRobots('noarchive', isChecked),
                    help: 'Do not show a "Cached" link in search results.'
                })
            )
        );
    };

    const SocialTab = ({ meta, updateMeta }) => {
        return el('div', { style: { marginTop: '15px' } },
            el(PanelBody, { title: 'Facebook (OpenGraph)', initialOpen: true },
                el(TextControl, {
                    label: 'Facebook Title',
                    value: meta['_serpforge_og_title'] || '',
                    onChange: (val) => updateMeta('_serpforge_og_title', val),
                    __next40pxDefaultSize: true,
                    __nextHasNoMarginBottom: true
                }),
                el(TextareaControl, {
                    label: 'Facebook Description',
                    value: meta['_serpforge_og_description'] || '',
                    onChange: (val) => updateMeta('_serpforge_og_description', val),
                    rows: 2,
                    __next40pxDefaultSize: true,
                    __nextHasNoMarginBottom: true
                }),
                el(TextControl, {
                    label: 'Image URL',
                    value: meta['_serpforge_og_image'] || '',
                    onChange: (val) => updateMeta('_serpforge_og_image', val),
                    help: 'Recommended size: 1200x630px',
                    __next40pxDefaultSize: true,
                    __nextHasNoMarginBottom: true
                })
            ),
            el(PanelBody, { title: 'Twitter Card', initialOpen: false },
                el(TextControl, {
                    label: 'Twitter Title',
                    value: meta['_serpforge_twitter_title'] || '',
                    onChange: (val) => updateMeta('_serpforge_twitter_title', val),
                    __next40pxDefaultSize: true,
                    __nextHasNoMarginBottom: true
                }),
                el(TextareaControl, {
                    label: 'Twitter Description',
                    value: meta['_serpforge_twitter_description'] || '',
                    onChange: (val) => updateMeta('_serpforge_twitter_description', val),
                    rows: 2,
                    __next40pxDefaultSize: true,
                    __nextHasNoMarginBottom: true
                })
            )
        );
    };

    const SchemaTab = ({ meta, updateMeta }) => {
        return el('div', { style: { marginTop: '15px' } },
            el(PanelBody, { title: 'Structured Data', initialOpen: true },
                el(SelectControl, {
                    label: 'Schema Type',
                    value: meta['_serpforge_schema_type'] || 'Article',
                    options: [
                        { label: 'Article', value: 'Article' },
                        { label: 'News Article', value: 'NewsArticle' },
                        { label: 'Blog Posting', value: 'BlogPosting' },
                        { label: 'Product', value: 'Product' },
                        { label: 'Service', value: 'Service' }
                    ],
                    onChange: (val) => updateMeta('_serpforge_schema_type', val),
                    __next40pxDefaultSize: true,
                    __nextHasNoMarginBottom: true
                }),
                el(TextControl, {
                    label: 'Headline',
                    value: meta['_serpforge_schema_headline'] || '',
                    onChange: (val) => updateMeta('_serpforge_schema_headline', val),
                    help: 'Overrides main title for schema.',
                    __next40pxDefaultSize: true,
                    __nextHasNoMarginBottom: true
                })
            )
        );
    };

    // --- MAIN PLUGIN COMPONENT ---

    const SERPforgeSEO = () => {
        const { postType, title, content, meta, permalink } = useSelect(select => {
            return {
                postType: select('core/editor').getCurrentPostType(),
                title: select('core/editor').getEditedPostAttribute('title'),
                content: select('core/editor').getEditedPostAttribute('content'),
                meta: select('core/editor').getEditedPostAttribute('meta') || {},
                permalink: select('core/editor').getPermalink(),
            };
        });

        const { editPost } = useDispatch('core/editor');

        const updateMeta = (key, value) => {
            editPost({
                meta: {
                    ...meta,
                    [key]: value
                }
            });
        };

        const [score, setScore] = useState(0);
        const [checks, setChecks] = useState([]);

        // Analysis Logic (Effect)
        useEffect(() => {
            const keyword = meta['_serpforge_focus_keyword'] || '';
            runAnalysis(title, content, keyword);
        }, [title, content, meta['_serpforge_focus_keyword']]);

        const runAnalysis = (docTitle, docContent, keyword) => {
            let tempScore = 0;
            let tempChecks = [];

            // Helper to strip tags and get clean text
            const getCleanText = (html) => {
                const doc = new DOMParser().parseFromString(html, 'text/html');
                return doc.body.textContent || '';
            };

            const cleanContent = docContent ? getCleanText(docContent) : '';
            const cleanTitle = docTitle ? docTitle : '';

            if (!keyword) {
                tempChecks.push({ id: 'no_kw', label: 'Set a Focus Keyword', status: 'info' });
            } else {
                const keywordLower = keyword.toLowerCase();
                const contentLower = cleanContent.toLowerCase();
                const titleLower = cleanTitle.toLowerCase();

                // 1. Keyword in Title
                if (titleLower.includes(keywordLower)) {
                    tempScore += 20;
                    tempChecks.push({ id: 'kw_title', label: 'Keyword in Title', status: 'good' });
                } else {
                    tempChecks.push({ id: 'kw_title', label: 'Keyword not in Title', status: 'critical' });
                }

                // 2. Keyword in Content
                if (contentLower.includes(keywordLower)) {
                    tempScore += 20;
                    tempChecks.push({ id: 'kw_content', label: 'Keyword in Content', status: 'good' });
                } else {
                    tempChecks.push({ id: 'kw_content', label: 'Keyword not in Content', status: 'critical' });
                }

                // 3. Keyword Density
                const words = contentLower.match(/\b\w+\b/g) || [];
                const wordCount = words.length;
                if (wordCount > 0) {
                    const matches = contentLower.split(keywordLower).length - 1;
                    const density = (matches / wordCount) * 100;
                    const densityFormatted = density.toFixed(2);

                    if (density >= 0.5 && density <= 2.5) {
                        tempScore += 10;
                        tempChecks.push({ id: 'kw_density', label: `Keyword Density: ${densityFormatted}% (Good)`, status: 'good' });
                    } else if (density > 2.5) {
                        tempChecks.push({ id: 'kw_density', label: `Keyword Density: ${densityFormatted}% (Too High)`, status: 'critical' });
                    } else {
                        tempChecks.push({ id: 'kw_density', label: `Keyword Density: ${densityFormatted}% (Too Low)`, status: 'improve' });
                    }
                }

                // 4. Keyword in Introduction (First 200 chars approximately)
                const introText = contentLower.substring(0, 200);
                if (introText.includes(keywordLower)) {
                    tempScore += 10;
                    tempChecks.push({ id: 'kw_intro', label: 'Keyword in Introduction', status: 'good' });
                } else {
                    tempChecks.push({ id: 'kw_intro', label: 'Keyword not in Introduction', status: 'improve' });
                }
            }

            // Word Count
            const wordCount = cleanContent.split(/\s+/).length;
            if (wordCount > 300) {
                tempScore += 10;
                tempChecks.push({ id: 'len', label: `Content Length: ${wordCount} words`, status: 'good' });
            } else {
                tempChecks.push({ id: 'len', label: `Content Length: ${wordCount} words (Target: 300+)`, status: 'improve' });
            }

            // Title Length
            if (cleanTitle.length >= 30 && cleanTitle.length <= 60) {
                tempScore += 10;
                tempChecks.push({ id: 'title_len', label: 'Title Length: Good', status: 'good' });
            } else {
                tempChecks.push({ id: 'title_len', label: `Title Length: ${cleanTitle.length} (Target: 30-60)`, status: 'improve' });
            }

            // Headings Check (H2/H3)
            const headingsRaw = (docContent.match(/<h[23][^>]*>/g) || []).length;
            if (headingsRaw > 0) {
                tempScore += 10;
                tempChecks.push({ id: 'headings', label: 'Subheadings (H2/H3) present', status: 'good' });
            } else {
                tempChecks.push({ id: 'headings', label: 'Use H2 or H3 subheadings', status: 'improve' });
            }

            // Readability: Flesch Reading Ease
            // Formula: 206.835 - 1.015(total words / total sentences) - 84.6(total syllables / total words)
            const countSyllables = (word) => {
                word = word.toLowerCase();
                if (word.length <= 3) return 1;
                word = word.replace(/(?:[^laeiouy]es|ed|[^laeiouy]e)$/, '');
                word = word.replace(/^y/, '');
                const syl = word.match(/[aeiouy]{1,2}/g);
                return syl ? syl.length : 1;
            };

            const sentences = cleanContent.split(/[.?!]+/).filter(s => s.trim().length > 0).length || 1;
            // cleanContent words
            const wordsMatch = cleanContent.match(/\b\w+\b/g) || [];
            const totalWords = wordsMatch.length;

            if (totalWords > 10) {
                let totalSyllables = 0;
                wordsMatch.forEach(w => totalSyllables += countSyllables(w));

                const flesch = 206.835 - 1.015 * (totalWords / sentences) - 84.6 * (totalSyllables / totalWords);
                const fleschScore = Math.min(100, Math.max(0, flesch)); // Clamp 0-100

                if (fleschScore >= 60) {
                    tempScore += 10;
                    tempChecks.push({ id: 'readability', label: `Readability: Easy (${fleschScore.toFixed(0)})`, status: 'good' });
                } else if (fleschScore >= 30) {
                    tempChecks.push({ id: 'readability', label: `Readability: Moderate (${fleschScore.toFixed(0)})`, status: 'improve' });
                } else {
                    tempChecks.push({ id: 'readability', label: `Readability: Difficult (${fleschScore.toFixed(0)})`, status: 'critical' });
                }
            }

            // --- AI SUGGESTIONS ---

            // 1. Power Words (Sentiment/Emotion)
            const powerWords = ['ultimate', 'guide', 'best', 'top', 'review', 'exclusive', 'secret', 'amazing', 'essential', 'proven', 'boost', 'growth', 'free', 'easy', 'simple', 'fast', 'secure', 'expert', 'strategy', 'tips'];
            const powerWordMatches = wordsMatch.filter(w => powerWords.includes(w.toLowerCase()));

            if (powerWordMatches.length > 0) {
                tempScore += 5;
                tempChecks.push({ id: 'ai_power', label: `AI Suggestion: Great usage of Power Words (${powerWordMatches.length} found)`, status: 'good' });
            } else {
                tempChecks.push({ id: 'ai_power', label: 'AI Suggestion: Use "Power Words" like "Ultimate", "Best", "Guide" to increase CTR.', status: 'improve' });
            }

            // 2. Paragraph Length Analysis
            const paragraphs = cleanContent.split(/\n\s*\n/);
            const longParagraphs = paragraphs.filter(p => p.split(/\s+/).length > 150);

            if (longParagraphs.length === 0) {
                tempScore += 5;
                tempChecks.push({ id: 'ai_para', label: 'AI Suggestion: Paragraph length looks good.', status: 'good' });
            } else {
                tempChecks.push({ id: 'ai_para', label: `AI Suggestion: ${longParagraphs.length} paragraph(s) are too long (>150 words). Break them up.`, status: 'improve' });
            }

            // Permalink Check (Basic) - If we have the permalink prop available
            // Note: We might need to pass permalink into this function if we want to check it.
            // For now, let's stick to content analysis.

            // Cap Score
            if (tempScore > 100) tempScore = 100;

            setScore(tempScore);
            setChecks(tempChecks);
        };

        const tabConfig = [
            { name: 'general', title: 'General', className: 'serpforge-tab-general' },
            { name: 'advanced', title: 'Advanced', className: 'serpforge-tab-advanced' },
            { name: 'social', title: 'Social', className: 'serpforge-tab-social' },
            { name: 'schema', title: 'Schema', className: 'serpforge-tab-schema' },
        ];

        return el(Fragment, {},
            el(ScoreCard, { score }),
            el(TabPanel, {
                className: 'serpforge-sidebar-tabs',
                activeClass: 'active-tab',
                tabs: tabConfig
            },
                (tab) => {
                    switch (tab.name) {
                        case 'general':
                            return el(GeneralTab, { meta, updateMeta, title, permalink, checks });
                        case 'advanced':
                            return el(AdvancedTab, { meta, updateMeta });
                        case 'social':
                            return el(SocialTab, { meta, updateMeta });
                        case 'schema':
                            return el(SchemaTab, { meta, updateMeta });
                        default: return null;
                    }
                })
        );
    };

    registerPlugin('serpforge-seo-sidebar', {
        icon: SERPFORGE_ICON,
        render: () => {
            return el(Fragment, {},
                el(PluginSidebarMoreMenuItem, {
                    target: 'serpforge-seo-sidebar',
                    icon: SERPFORGE_ICON
                }, 'SERPforge SEO'),
                el(PluginSidebar, {
                    name: 'serpforge-seo-sidebar',
                    title: 'SERPforge SEO',
                    icon: SERPFORGE_ICON
                }, el(SERPforgeSEO))
            );
        }
    });

})(window.wp);
