<?php
namespace SerpForge\Admin;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Handles the Admin Dashboard and UI.
 */
class AdminManager
{
    private $settings;
    private $modules;

    /**
     * Constructor.
     */
    public function __construct()
    {
        $this->settings = new SettingsController();
        $this->modules = new ModuleSettingsController();

        add_action('admin_menu', [$this, 'register_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
        add_action('enqueue_block_editor_assets', [$this, 'enqueue_editor_assets']);
        add_action('wp_ajax_serpforge_toggle_module', [$this, 'toggle_module']);
        add_action('wp_ajax_serpforge_save_bulk_meta', [$this->modules, 'save_bulk_meta']);
    }

    /**
     * AJAX: Toggle module state.
     */
    public function toggle_module()
    {
        check_ajax_referer('serpforge_admin_nonce', 'nonce');

        $module = isset($_POST['module']) ? sanitize_text_field($_POST['module']) : '';
        if (!$module) {
            wp_send_json_error();
        }

        $current_modules = get_option('serpforge_active_modules', []);

        if (in_array($module, $current_modules)) {
            $current_modules = array_values(array_diff($current_modules, [$module]));
        } else {
            $current_modules[] = $module;
            $current_modules = array_values($current_modules); // Ensure sequential
        }

        update_option('serpforge_active_modules', $current_modules);
        wp_send_json_success(['active' => $current_modules]);
    }

    /**
     * Register the plugin admin menu.
     */
    public function register_admin_menu()
    {
        add_menu_page(
            __('SERPforge SEO', 'serpforge-seo'),
            __('SERPforge SEO', 'serpforge-seo'),
            'manage_options',
            'serpforge',
            [$this, 'render_dashboard'],
            'dashicons-chart-line',
            30
        );

        add_submenu_page(
            'serpforge',
            __('Dashboard', 'serpforge-seo'),
            __('Dashboard', 'serpforge-seo'),
            'manage_options',
            'serpforge',
            [$this, 'render_dashboard']
        );

        // General Settings
        add_submenu_page(
            'serpforge',
            __('General Settings', 'serpforge-seo'),
            __('General Settings', 'serpforge-seo'),
            'manage_options',
            'serpforge-general',
            [$this->settings, 'render_general_settings']
        );
        add_submenu_page(
            'serpforge',
            __('Redirections', 'serpforge-seo'),
            __('Redirections', 'serpforge-seo'),
            'manage_options',
            'serpforge-redirects',
            [$this->modules, 'render_redirects_page']
        );
        add_submenu_page(
            'serpforge',
            __('Image SEO', 'serpforge-seo'),
            __('Image SEO', 'serpforge-seo'),
            'manage_options',
            'serpforge-image-seo',
            [$this->settings, 'render_image_seo_settings']
        );
        add_submenu_page(
            'serpforge',
            __('Sitemap', 'serpforge-seo'),
            __('Sitemap', 'serpforge-seo'),
            'manage_options',
            'serpforge-sitemap',
            [$this->settings, 'render_sitemap_settings']
        );
        add_submenu_page(
            'serpforge',
            __('Bulk Editor', 'serpforge-seo'),
            __('Bulk Editor', 'serpforge-seo'),
            'manage_options',
            'serpforge-bulk-editor',
            [$this->modules, 'render_bulk_editor']
        );

        // Settings pages for other modules
        add_submenu_page('serpforge', __('WooCommerce', 'serpforge-seo'), __('WooCommerce', 'serpforge-seo'), 'manage_options', 'serpforge-woocommerce', [$this->modules, 'render_woocommerce_settings']);
        add_submenu_page('serpforge', __('Local SEO', 'serpforge-seo'), __('Local SEO', 'serpforge-seo'), 'manage_options', 'serpforge-local-seo', [$this->modules, 'render_local_seo_settings']);
        add_submenu_page('serpforge', __('404 Monitor', 'serpforge-seo'), __('404 Monitor', 'serpforge-seo'), 'manage_options', 'serpforge-404-monitor', [$this->modules, 'render_404_logs']);
        add_submenu_page('serpforge', __('Schema', 'serpforge-seo'), __('Schema', 'serpforge-seo'), 'manage_options', 'serpforge-schema', [$this->modules, 'render_schema_settings']);
        add_submenu_page('serpforge', __('Link Counter', 'serpforge-seo'), __('Link Counter', 'serpforge-seo'), 'manage_options', 'serpforge-link-counter', [$this->modules, 'render_link_counter_settings']);
        add_submenu_page('serpforge', __('News Sitemap', 'serpforge-seo'), __('News Sitemap', 'serpforge-seo'), 'manage_options', 'serpforge-news-sitemap', [$this->modules, 'render_news_sitemap_settings']);
        add_submenu_page('serpforge', __('Podcast', 'serpforge-seo'), __('Podcast', 'serpforge-seo'), 'manage_options', 'serpforge-podcast', [$this->modules, 'render_podcast_settings']);
    }

    /**
     * Enqueue Admin CSS and JS.
     *
     * @param string $hook Current admin page hook.
     */
    public function enqueue_assets($hook)
    {
        if (strpos($hook, 'serpforge') === false) {
            return;
        }

        wp_enqueue_style(
            'serpforge-admin-css',
            SERPFORGE_URL . 'assets/css/admin-style.css',
            [],
            SERPFORGE_VERSION
        );

        wp_enqueue_script(
            'serpforge-admin-js',
            SERPFORGE_URL . 'assets/js/admin-app.js',
            ['jquery', 'wp-element'], // wp-element for React if we go that route
            SERPFORGE_VERSION . time(),
            true
        );

        wp_localize_script('serpforge-admin-js', 'serpforgeData', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('serpforge_admin_nonce'),
            'activeModules' => array_values(get_option('serpforge_active_modules', ['image-seo', 'sitemap', 'link-counter', 'schema'])),
        ]);

    }

    /**
     * Enqueue Block Editor (Gutenberg) Assets.
     */
    public function enqueue_editor_assets()
    {
        wp_enqueue_script(
            'serpforge-block-editor-js',
            SERPFORGE_URL . 'assets/js/block-editor.js',
            ['wp-plugins', 'wp-edit-post', 'wp-element', 'wp-components', 'wp-data', 'wp-compose', 'wp-editor'],
            SERPFORGE_VERSION . time(), // Cache busting
            true
        );
    }

    /**
     * Render the main dashboard page.
     */
    public function render_dashboard()
    {
        // Check for Active License to hide upsell
        if (!\SerpForge\License\LicenseManager::is_active()) {
            ?>
            <div class="serpforge-upsell-banner">
                <div class="upsell-content">
                    <h2>🚀 Go Pro & Dominate Search!</h2>
                    <p>Unlock <strong>WooCommerce SEO</strong>, <strong>Local SEO</strong>, <strong>404 Monitoring</strong>, and the
                        <strong>Bulk Editor</strong>.
                    </p>
                </div>
                <div class="upsell-action">
                    <a href="https://serpforge.compalgolabs.com/checkout.html" target="_blank"
                        class="button button-primary button-hero">Upgrade to
                        Pro - $49</a>
                </div>
            </div>
            <?php
        }

        echo '<div id="serpforge-admin-app"></div>';
        // Fallback content if JS fails
        echo '<noscript>' . esc_html__('Please enable JavaScript to use SERPforge SEO.', 'serpforge-seo') . '</noscript>';
    }
}
