<?php
namespace SerpForge\Admin;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Handles Post Editor Metaboxes.
 */
class Metabox
{

    public function __construct()
    {
        add_action('add_meta_boxes', [$this, 'add_seo_metabox']);
        add_action('save_post', [$this, 'save_seo_meta']);
    }

    /**
     * Register the metabox.
     */
    public function add_seo_metabox()
    {
        $screens = ['post', 'page']; // Extendable for CPTs
        foreach ($screens as $screen) {
            add_meta_box(
                'serpforge_seo_box',
                __('SERPforge SEO', 'serpforge-seo'),
                [$this, 'render_metabox'],
                $screen,
                'normal',
                'high'
            );
        }
    }

    /**
     * Render the metabox content.
     *
     * @param \WP_Post $post
     */
    public function render_metabox($post)
    {
        wp_nonce_field('serpforge_save_seo_data', 'serpforge_seo_nonce');

        $seo_title = get_post_meta($post->ID, '_serpforge_seo_title', true);
        $seo_desc = get_post_meta($post->ID, '_serpforge_seo_description', true);
        $robots = get_post_meta($post->ID, '_serpforge_robots', true);
        $keyword = get_post_meta($post->ID, '_serpforge_focus_keyword', true);

        if (!is_array($robots)) {
            $robots = [];
        }

        // Enqueue our specific JS for this (Assuming admin-app.js includes it or new file)
        // For MVP, we will keep inline or standard admin JS. 
        // Let's rely on admin-app.js being loaded (checked in AdminManager).
        ?>
        <div class="serpforge-metabox-wrapper">
            <!-- Tabs -->
            <div class="serpforge-tabs">
                <a href="#" class="serpforge-tab active" data-tab="general"><span class="dashicons dashicons-admin-home"></span>
                    General</a>
                <a href="#" class="serpforge-tab" data-tab="advanced"><span class="dashicons dashicons-admin-settings"></span>
                    Advanced</a>
                <a href="#" class="serpforge-tab" data-tab="social"><span class="dashicons dashicons-share"></span> Social</a>
                <a href="#" class="serpforge-tab" data-tab="schema"><span class="dashicons dashicons-schema"></span> Schema</a>
            </div>

            <div class="serpforge-tab-content active" id="tab-general">

                <!-- Snippet Preview -->
                <div class="serpforge-snippet-preview">
                    <h4>Snippet Preview</h4>
                    <div class="preview-container">
                        <div class="preview-url"><?php echo esc_url(get_permalink($post->ID)); ?></div>
                        <div class="preview-title">
                            <?php echo $seo_title ? esc_html($seo_title) : esc_html($post->post_title); ?>
                        </div>
                        <div class="preview-desc">
                            <?php echo $seo_desc ? esc_html($seo_desc) : 'Please provide a meta description.'; ?>
                        </div>
                    </div>
                    <button type="button" class="button button-small" id="edit-snippet-btn">Edit Snippet</button>

                    <div class="snippet-editor hidden">
                        <div style="margin-bottom: 15px;">
                            <button type="button" class="button button-secondary" id="serpforge-ai-generate">
                                ✨ Generate with AI
                            </button>
                            <span class="spinner" id="serpforge-ai-spinner" style="float:none;"></span>
                        </div>

                        <div class="serpforge-field">
                            <label>SEO Title</label>
                            <input type="text" id="serpforge_seo_title" name="serpforge_seo_title"
                                value="<?php echo esc_attr($seo_title); ?>" placeholder="%title% %sep% %sitename%">
                            <div class="char-counter"><span id="title-char-count">0</span>/60</div>
                        </div>
                        <div class="serpforge-field">
                            <label>Meta Description</label>
                            <textarea id="serpforge_seo_description" name="serpforge_seo_description"
                                rows="2"><?php echo esc_textarea($seo_desc); ?></textarea>
                            <div class="char-counter"><span id="desc-char-count">0</span>/160</div>
                        </div>
                    </div>
                </div>

                <!-- Focus Keyword -->
                <div class="serpforge-field focus-keyword-section">
                    <label>Focus Keyword</label>
                    <input type="text" id="serpforge_focus_keyword" name="serpforge_focus_keyword"
                        value="<?php echo esc_attr($keyword); ?>" placeholder="Insert focus keyword">
                </div>

                <!-- Analysis Results -->
                <div class="serpforge-analysis-section">
                    <div class="score-dial" id="serpforge-score-dial">
                        <span id="serpforge-score-val">--</span>/100
                    </div>
                    <div class="analysis-list" id="serpforge-analysis-results">
                        <p>Update post to see analysis...</p>
                    </div>
                </div>

            </div>

            <div class="serpforge-tab-content" id="tab-advanced">
                <div class="serpforge-field">
                    <label>Robots Meta</label>
                    <label><input type="checkbox" name="serpforge_robots[]" value="noindex" <?php checked(in_array('noindex', $robots)); ?>> No Index</label>
                    <label><input type="checkbox" name="serpforge_robots[]" value="nofollow" <?php checked(in_array('nofollow', $robots)); ?>> No Follow</label>
                    <label><input type="checkbox" name="serpforge_robots[]" value="noarchive" <?php checked(in_array('noarchive', $robots)); ?>> No Archive</label>
                </div>
            </div>

            <div class="serpforge-tab-content" id="tab-social">
                <h4>Facebook (OpenGraph)</h4>
                <div class="serpforge-field">
                    <label>Facebook Title</label>
                    <input type="text" name="serpforge_og_title"
                        value="<?php echo esc_attr(get_post_meta($post->ID, '_serpforge_og_title', true)); ?>"
                        placeholder="Title for Facebook">
                </div>
                <div class="serpforge-field">
                    <label>Facebook Description</label>
                    <textarea name="serpforge_og_description" rows="2"
                        placeholder="Description for Facebook"><?php echo esc_textarea(get_post_meta($post->ID, '_serpforge_og_description', true)); ?></textarea>
                </div>
                <div class="serpforge-field">
                    <label>Facebook Image URL</label>
                    <input type="text" name="serpforge_og_image"
                        value="<?php echo esc_attr(get_post_meta($post->ID, '_serpforge_og_image', true)); ?>"
                        placeholder="Image URL for Facebook">
                </div>

                <hr>

                <h4>Twitter Card</h4>
                <div class="serpforge-field">
                    <label>Twitter Title</label>
                    <input type="text" name="serpforge_twitter_title"
                        value="<?php echo esc_attr(get_post_meta($post->ID, '_serpforge_twitter_title', true)); ?>"
                        placeholder="Title for Twitter">
                </div>
                <div class="serpforge-field">
                    <label>Twitter Description</label>
                    <textarea name="serpforge_twitter_description" rows="2"
                        placeholder="Description for Twitter"><?php echo esc_textarea(get_post_meta($post->ID, '_serpforge_twitter_description', true)); ?></textarea>
                </div>
                <div class="serpforge-field">
                    <label>Twitter Image URL</label>
                    <input type="text" name="serpforge_twitter_image"
                        value="<?php echo esc_attr(get_post_meta($post->ID, '_serpforge_twitter_image', true)); ?>"
                        placeholder="Image URL for Twitter">
                </div>
            </div>

            <div class="serpforge-tab-content" id="tab-schema">
                <div class="serpforge-field">
                    <label>Schema Type</label>
                    <select name="serpforge_schema_type" id="serpforge_schema_type">
                        <?php
                        $current_type = get_post_meta($post->ID, '_serpforge_schema_type', true) ?: 'Article';
                        $types = [
                            'Article' => 'Article',
                            'NewsArticle' => 'News Article',
                            'BlogPosting' => 'Blog Posting',
                            'Product' => 'Product',
                            'Service' => 'Service',
                            'Review' => 'Review',
                            'WebPage' => 'Web Page'
                        ];
                        foreach ($types as $val => $label) {
                            echo '<option value="' . esc_attr($val) . '" ' . selected($current_type, $val, false) . '>' . esc_html($label) . '</option>';
                        }
                        ?>
                    </select>
                    <p class="description">Select the Rich Snippet type for this content.</p>
                </div>

                <div class="serpforge-field">
                    <label>Headline (Optional)</label>
                    <input type="text" name="serpforge_schema_headline"
                        value="<?php echo esc_attr(get_post_meta($post->ID, '_serpforge_schema_headline', true)); ?>"
                        placeholder="Override main title">
                </div>

                <div class="serpforge-field">
                    <label>Description (Optional)</label>
                    <textarea name="serpforge_schema_description" rows="2"
                        placeholder="Override main description"><?php echo esc_textarea(get_post_meta($post->ID, '_serpforge_schema_description', true)); ?></textarea>
                </div>
            </div>

        </div>
        <?php
    }

    /**
     * Save post meta.
     *
     * @param int $post_id
     */
    public function save_seo_meta($post_id)
    {
        if (!isset($_POST['serpforge_seo_nonce'])) {
            return;
        }
        if (!wp_verify_nonce($_POST['serpforge_seo_nonce'], 'serpforge_save_seo_data')) {
            return;
        }
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        $fields = [
            'serpforge_seo_title',
            'serpforge_seo_description',
            'serpforge_focus_keyword',
            'serpforge_og_title',
            'serpforge_og_description',
            'serpforge_og_image',
            'serpforge_twitter_title',
            'serpforge_twitter_description',
            'serpforge_twitter_image',
            'serpforge_schema_type',
            'serpforge_schema_headline',
            'serpforge_schema_description'
        ];

        foreach ($fields as $field) {
            if (isset($_POST[$field])) {
                update_post_meta($post_id, '_' . $field, sanitize_text_field($_POST[$field]));
            }
        }

        // Robots (Array)
        if (isset($_POST['serpforge_robots'])) {
            $robots = array_map('sanitize_text_field', $_POST['serpforge_robots']);
            update_post_meta($post_id, '_serpforge_robots', $robots);
        } else {
            delete_post_meta($post_id, '_serpforge_robots');
        }
    }
}
