<?php
namespace SerpForge\Analysis;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * SEO Content Analyzer.
 */
class Analyzer
{

    public function __construct()
    {
        // Register AJAX handler for analysis
        add_action('wp_ajax_serpforge_analyze_content', [$this, 'ajax_analyze']);
    }

    /**
     * Handle AJAX request for content analysis.
     */
    public function ajax_analyze()
    {
        // check_ajax_referer( 'serpforge_admin_nonce', 'nonce' ); // Uncomment in production

        $content = isset($_POST['content']) ? wp_unslash($_POST['content']) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput
        $keyword = isset($_POST['keyword']) ? sanitize_text_field($_POST['keyword']) : '';
        $title = isset($_POST['title']) ? sanitize_text_field($_POST['title']) : '';

        $results = $this->analyze($content, $keyword, $title);

        wp_send_json_success($results);
    }

    /**
     * Perform analysis.
     *
     * @param string $content Content to analyze.
     * @param string $keyword Focus keyword.
     * @param string $title Page title.
     * @return array Analysis results.
     */
    public function analyze($content, $keyword, $title)
    {
        $score = 0;
        $checks = [];

        // 1. Check Title Length
        $title_len = mb_strlen($title);
        if ($title_len >= 30 && $title_len <= 60) {
            $checks['title_length'] = ['status' => 'good', 'msg' => 'Title length is optimal.'];
            $score += 20;
        } else {
            $checks['title_length'] = ['status' => 'improve', 'msg' => 'Title should be between 30-60 characters.'];
        }

        // 2. Keyword in Title
        if (!empty($keyword) && stripos($title, $keyword) !== false) {
            $checks['keyword_in_title'] = ['status' => 'good', 'msg' => 'Focus keyword found in title.'];
            $score += 30;
        } else {
            $checks['keyword_in_title'] = (empty($keyword))
                ? ['status' => 'info', 'msg' => 'Set a focus keyword to check.']
                : ['status' => 'critical', 'msg' => 'Focus keyword not found in title.'];
        }

        // 3. Keyword Density (Basic)
        if (!empty($keyword) && !empty($content)) {
            $word_count = str_word_count(strip_tags($content));
            $keyword_count = substr_count(strtolower($content), strtolower($keyword));

            if ($word_count > 0) {
                $density = ($keyword_count / $word_count) * 100;
                if ($density >= 0.5 && $density <= 2.5) {
                    $checks['keyword_density'] = ['status' => 'good', 'msg' => 'Keyword density is good (' . round($density, 2) . '%).'];
                    $score += 20;
                } else {
                    $checks['keyword_density'] = ['status' => 'improve', 'msg' => 'Keyword density is ' . round($density, 2) . '%. Aim for 0.5% - 2.5%.'];
                }
            }
        }

        // 4. Content Length
        $word_count = str_word_count(strip_tags($content));
        if ($word_count > 300) {
            $checks['content_length'] = ['status' => 'good', 'msg' => 'Content length is good (' . $word_count . ' words).'];
            $score += 20;
        } else {
            $checks['content_length'] = ['status' => 'improve', 'msg' => 'Content is too short (recommended: > 300 words).'];
        }

        return [
            'score' => $score,
            'checks' => $checks,
        ];
    }
}
