<?php
declare(strict_types=1);

namespace SerpForge\Core;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Main Plugin Class.
 */
final class Plugin
{

    /**
     * Glance instance.
     *
     * @var Plugin
     */
    private static $instance;

    /**
     * Main Instance.
     *
     * Ensures only one instance of the plugin class is loaded or can be loaded.
     *
     * @return Plugin - Main instance.
     */
    public static function instance()
    {
        if (!isset(self::$instance) && !(self::$instance instanceof Plugin)) {
            self::$instance = new Plugin();
            self::$instance->setup_constants();
            self::$instance->includes();
            self::$instance->init_hooks();
        }
        return self::$instance;
    }

    /**
     * Throw error on object clone.
     *
     * The whole idea of the singleton design pattern is that there is a single
     * object therefore, we don't want the object to be cloned.
     */
    public function __clone()
    {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'serpforge-seo'), '1.0.0');
    }

    /**
     * Disable unserializing of the class.
     */
    public function __wakeup()
    {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'serpforge-seo'), '1.0.0');
    }

    /**
     * Setup plugin constants.
     */
    private function setup_constants()
    {
        // Additional constants can be defined here if needed.
    }

    /**
     * Include required files and modules.
     */
    private function includes()
    {
        // Initialize Modules

        // 1. Core / Admin
        if (is_admin()) {
            new \SerpForge\Admin\AdminManager();
            new \SerpForge\Admin\Metabox();
            new \SerpForge\Admin\DashboardData();
        }

        // Get Active Modules
        $active_modules = get_option('serpforge_active_modules', [
            'image-seo',
            'sitemap',
            'link-counter',
            'schema'
        ]);



        // 2. Free Modules
        new \SerpForge\Meta\MetaManager(); // Core Meta tags (Always On)
        new \SerpForge\License\LicenseManager(); // License Handler (Always On)

        if (in_array('sitemap', $active_modules)) {
            new \SerpForge\Sitemap\SitemapGenerator();
        }
        if (in_array('schema', $active_modules)) {
            new \SerpForge\Schema\SchemaManager();
        }
        if (in_array('image-seo', $active_modules)) {
            new \SerpForge\Image\ImageManager();
        }

        new \SerpForge\Analysis\Analyzer();            // Content Analysis (Always On for now, or toggle?)

        if (in_array('link-counter', $active_modules)) {
            new \SerpForge\LinkCounter\LinkCounter();
        }

        if (wp_doing_ajax()) {
            new \SerpForge\AI\AISuggestions();
        }

        // Elementor Support
        if (did_action('elementor/loaded')) {
            new \SerpForge\Elementor\ElementorManager();
        }

        // 3. Pro Modules (Gated by License)
        if (\SerpForge\License\LicenseManager::is_active()) {

            // WooCommerce SEO
            if (in_array('woocommerce', $active_modules) && class_exists('\SerpForge\WooCommerce\ProductManager')) {
                new \SerpForge\WooCommerce\ProductManager();
            }

            // Local SEO
            if (in_array('local-seo', $active_modules) && class_exists('\SerpForge\Local\LocalManager')) {
                new \SerpForge\Local\LocalManager();
            }

            // 404 Monitor
            if (in_array('monitor', $active_modules) && class_exists('\SerpForge\Monitor\FourOhFourMonitor')) {
                new \SerpForge\Monitor\FourOhFourMonitor();
            }

            // Redirections
            if (in_array('redirects', $active_modules) && class_exists('\SerpForge\Redirect\RedirectManager')) {
                new \SerpForge\Redirect\RedirectManager();
            }

            // News Sitemap
            if (in_array('news-sitemap', $active_modules) && class_exists('\SerpForge\Sitemap\NewsSitemap')) {
                new \SerpForge\Sitemap\NewsSitemap();
            }

            // Podcast Module
            if (in_array('podcast', $active_modules) && class_exists('\SerpForge\Podcast\PodcastManager')) {
                new \SerpForge\Podcast\PodcastManager();
            }

        }
    }

    /**
     * Hook into actions and filters.
     */
    private function init_hooks()
    {
        register_activation_hook(SERPFORGE_FILE, [$this, 'activate']);
        register_deactivation_hook(SERPFORGE_FILE, [$this, 'deactivate']);

        add_action('plugins_loaded', [$this, 'on_plugins_loaded']);
        add_action('init', [$this, 'register_meta_fields']);
    }

    /**
     * Register Post Meta for REST API.
     */
    public function register_meta_fields()
    {
        $post_types = ['post', 'page'];
        $meta_keys = [
            '_serpforge_focus_keyword',
            '_serpforge_seo_title',
            '_serpforge_seo_description',
            '_serpforge_og_title',
            '_serpforge_og_description',
            '_serpforge_og_image',
            '_serpforge_twitter_title',
            '_serpforge_twitter_description',
            '_serpforge_twitter_image',
            '_serpforge_schema_type',
            '_serpforge_schema_headline',
            '_serpforge_schema_description',
            '_serpforge_disable_image_seo'
        ];

        foreach ($post_types as $pt) {
            foreach ($meta_keys as $key) {
                register_post_meta($pt, $key, [
                    'show_in_rest' => true,
                    'single' => true,
                    'type' => 'string',
                    'auth_callback' => function () {
                        return current_user_can('edit_posts');
                    }
                ]);
            }

            // Robots is an array
            register_post_meta($pt, '_serpforge_robots', [
                'show_in_rest' => ['schema' => ['type' => 'array', 'items' => ['type' => 'string']]],
                'single' => true,
                'type' => 'array',
                'auth_callback' => function () {
                    return current_user_can('edit_posts');
                }
            ]);
        }
    }

    /**
     * Activation hook.
     */
    public function activate()
    {
        // Flush rewrite rules for sitemaps
        flush_rewrite_rules();

        // Schedule background link scan
        if (!wp_next_scheduled('serpforge_cron_link_scan')) {
            wp_schedule_single_event(time(), 'serpforge_cron_link_scan');
        }
    }

    /**
     * Deactivation hook.
     */
    public function deactivate()
    {
        flush_rewrite_rules();
    }

    /**
     * Plugins loaded action.
     */
    public function on_plugins_loaded()
    {
        // WordPress.org automatically loads translations for hosted plugins
    }
}
