<?php
namespace SerpForge\Elementor;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Elementor Integration Manager.
 */
class ElementorManager
{
    public function __construct()
    {
        // Hook into Elementor controls
        add_action('elementor/documents/register_controls', [$this, 'register_document_controls']);
    }

    /**
     * Register controls in the Page Settings panel.
     *
     * @param \Elementor\Core\DocumentTypes\PageBase $document
     */
    public function register_document_controls($document)
    {
        // Only valid for posts/pages that support our meta
        if (!$document instanceof \Elementor\Core\DocumentTypes\PageBase && !$document instanceof \Elementor\Modules\PageTemplates\Module) {
            return;
        }

        // Elementor uses its own control stack. 
        // We add a new section to the "Settings" tab.

        $document->start_controls_section(
            'serpforge_seo_section',
            [
                'label' => __('SERPforge SEO', 'serpforge-seo'),
                'tab' => \Elementor\Controls_Manager::TAB_SETTINGS,
            ]
        );

        $document->add_control(
            'serpforge_notice',
            [
                'type' => \Elementor\Controls_Manager::RAW_HTML,
                'raw' => '<div style="background:#f0f0f1; padding:10px; border-left:4px solid #2271b1; margin-bottom:10px;">' .
                    __('Edit your SEO metadata directly here. Frontend analysis is currently only available in the standard editor.', 'serpforge-seo') .
                    '</div>',
            ]
        );

        // SEO Title
        // Elementor doesn't map directly to post meta key automatically for all types perfectly, 
        // but 'dynamic' settings often do. 
        // However, for direct meta editing, we usually need to specify valid mapping or handle save manualy?
        // Actually, Elementor documents save settings to their own meta key structures usually (`_elementor_data`, etc).
        // BUT, Elementor PRO allows usage of "Page Settings" that map to WP options or meta.

        // Let's try the direct registration method.
        // It's safer to register these as standard controls and let Elementor handle them IF they are just settings.
        // But we want to sync with `_serpforge_seo_title`.

        // We will simple use the control, but we might need to hook into 'save' if Elementor doesn't auto-sync custom keys.
        // Elementor Page Settings are usually saved in `_elementor_page_settings`.
        // We might need to sync them back to our keys on save.

        $document->add_control(
            'serpforge_seo_title_input',
            [
                'label' => __('SEO Title', 'serpforge-seo'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'placeholder' => get_the_title($document->get_main_id()),
                'label_block' => true,
                'default' => get_post_meta($document->get_main_id(), '_serpforge_seo_title', true),
            ]
        );

        $document->add_control(
            'serpforge_seo_description_input',
            [
                'label' => __('Meta Description', 'serpforge-seo'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'rows' => 3,
                'label_block' => true,
                'default' => get_post_meta($document->get_main_id(), '_serpforge_seo_description', true),
            ]
        );

        $document->add_control(
            'serpforge_focus_keyword_input',
            [
                'label' => __('Focus Keyword', 'serpforge-seo'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'label_block' => true,
                'default' => get_post_meta($document->get_main_id(), '_serpforge_focus_keyword', true),
            ]
        );

        $document->end_controls_section();

        // Hook save
        add_action('elementor/document/after_save', [$this, 'save_document_settings'], 10, 2);
    }

    /**
     * Sync Elementor settings to native Post Meta.
     *
     * @param \Elementor\Core\Base\Document $document
     * @param array $data
     */
    public function save_document_settings($document, $data)
    {
        // Settings are in $document->get_settings('control_name');
        // But $data usually contains the raw updated settings too.

        $settings = $document->get_settings();

        if (isset($settings['serpforge_seo_title_input'])) {
            update_post_meta($document->get_main_id(), '_serpforge_seo_title', sanitize_text_field($settings['serpforge_seo_title_input']));
        }

        if (isset($settings['serpforge_seo_description_input'])) {
            update_post_meta($document->get_main_id(), '_serpforge_seo_description', sanitize_textarea_field($settings['serpforge_seo_description_input']));
        }

        if (isset($settings['serpforge_focus_keyword_input'])) {
            update_post_meta($document->get_main_id(), '_serpforge_focus_keyword', sanitize_text_field($settings['serpforge_focus_keyword_input']));
        }
    }
}
