<?php
namespace SerpForge\Image;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Handles Image SEO: Auto Alt and Auto Title attributes.
 */
class ImageManager
{
    public function __construct()
    {
        // Only run on frontend to modify output, or admin if we process on save (Frontend on-the-fly is safer/easier for now)
        if (!is_admin()) {
            add_filter('the_content', [$this, 'add_image_attributes'], 99);
        }
    }

    /**
     * Check if module is active.
     */
    private function is_active()
    {
        $modules = get_option('serpforge_active_modules', []);
        return in_array('image-seo', $modules);
    }

    /**
     * Filter content to add missing Alt and Title tags.
     */
    public function add_image_attributes($content)
    {
        if (!$this->is_active()) {
            return $content;
        }

        // Check Per-Post Disable logic
        global $post;
        if ($post && get_post_meta($post->ID, '_serpforge_disable_image_seo', true) === '1') {
            return $content;
        }

        // Parse HTML
        $dom = new \DOMDocument();
        // Suppress errors for malformed HTML
        libxml_use_internal_errors(true);
        // UTF-8 hack
        $dom->loadHTML(mb_convert_encoding($content, 'HTML-ENTITIES', 'UTF-8'), LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
        libxml_clear_errors();

        $images = $dom->getElementsByTagName('img');

        $modified = false;

        foreach ($images as $img) {
            $src = $img->getAttribute('src');
            $filename = pathinfo($src, PATHINFO_FILENAME);
            $clean_name = ucwords(str_replace(['-', '_'], ' ', $filename));

            // Fix Alt
            if (!$img->hasAttribute('alt') || $img->getAttribute('alt') === '') {
                $img->setAttribute('alt', $clean_name);
                $modified = true;
            }

            // Fix Title (Pro Only)
            if (\SerpForge\License\LicenseManager::is_active()) {
                if (!$img->hasAttribute('title') || $img->getAttribute('title') === '') {
                    $img->setAttribute('title', $clean_name);
                    $modified = true;
                }
            }
        }

        if ($modified) {
            return $dom->saveHTML();
        }

        return $content;
    }
}
