<?php
namespace SerpForge\LinkCounter;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Counts Internal and External links in posts.
 */
class LinkCounter
{
    public function __construct()
    {
        add_action('wp_ajax_serpforge_recount_links', [$this, 'ajax_recount']);

        // Count links on save
        add_action('save_post', [$this, 'count_links'], 10, 3);

        // Add Columns to Posts and Pages
        add_filter('manage_post_posts_columns', [$this, 'add_column']);
        add_filter('manage_page_posts_columns', [$this, 'add_column']);

        // Render Columns
        add_action('manage_post_posts_custom_column', [$this, 'render_column'], 10, 2);
        add_action('manage_page_posts_custom_column', [$this, 'render_column'], 10, 2);

        // Background Scan Hook
        add_action('serpforge_cron_link_scan', [$this, 'run_batch_scan']);
    }

    /**
     * Background Batch Scanner (Cron Job).
     * Scans posts that haven't been counted yet.
     */
    public function run_batch_scan()
    {
        $args = [
            'post_type' => ['post', 'page'],
            'post_status' => 'publish',
            'posts_per_page' => 20, // Small batch
            'fields' => 'ids',
            'meta_query' => [
                [
                    'key' => 'serpforge_internal_links',
                    'compare' => 'NOT EXISTS'
                ]
            ]
        ];

        $query = new \WP_Query($args);

        if ($query->have_posts()) {
            foreach ($query->posts as $post_id) {
                $this->process_post($post_id);
            }
            // Schedule next batch immediately
            wp_schedule_single_event(time() + 5, 'serpforge_cron_link_scan');
        }
    }

    /**
     * Hook callback for save_post.
     */
    public function count_links($post_id, $post, $update)
    {
        if (wp_is_post_revision($post_id)) {
            return;
        }
        $this->process_post($post_id);
    }

    /**
     * Process a single post to count links.
     */
    public function process_post($post_id)
    {
        $post = get_post($post_id);
        if (!$post)
            return;

        $content = $post->post_content;

        // Count Internal (Relative or includes site URL)
        $home_url = preg_quote(home_url(), '/');
        $internal_pattern = '/href=["\'](' . $home_url . '|\/)[^"\']*["\']/';
        preg_match_all($internal_pattern, $content, $matches);
        $internal_count = count($matches[0]);

        // Count External (http/https but not home_url)
        // Simple approx regex to find http/https links that do NOT match home_url
        // Simplest way: Find all http links, subtract those that contain home_url
        $all_link_pattern = '/href=["\'](http[s]?:\/\/[^"\']+)["\']/';
        preg_match_all($all_link_pattern, $content, $all_matches);

        // Filter out internal stats from "all matches" if utilizing strict regex is hard
        // But for MVP the previous logic was: external = total - internal. 
        // Note: The previous logic assumed all internal links were caught by $internal_pattern
        // which catches relative and home_url.
        // So Total Http Links might include home_url links.

        $total_links_found = count($all_matches[0]);
        // Let's refine: External is strictly domains that are not this domain.
        $external_count = 0;
        $domain = parse_url(home_url(), PHP_URL_HOST);

        if (!empty($all_matches[1])) {
            foreach ($all_matches[1] as $url) {
                $host = parse_url($url, PHP_URL_HOST);
                if ($host && strpos($host, $domain) === false) {
                    $external_count++;
                }
            }
        }

        update_post_meta($post_id, 'serpforge_internal_links', $internal_count);
        update_post_meta($post_id, 'serpforge_external_links', $external_count);
    }

    /**
     * AJAX: Recount all posts.
     */
    public function ajax_recount()
    {
        check_ajax_referer('serpforge_admin_nonce', 'nonce');

        if (!current_user_can('edit_posts')) {
            wp_send_json_error('Permission denied');
        }

        // Get all published posts and pages
        $args = [
            'post_type' => ['post', 'page'],
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'fields' => 'ids'
        ];

        $query = new \WP_Query($args);
        $ids = $query->posts;
        $count = 0;

        foreach ($ids as $id) {
            $this->process_post($id);
            $count++;
        }

        wp_send_json_success(['processed' => $count]);
    }

    public function add_column($columns)
    {
        $columns['serpforge_links'] = 'Links';
        return $columns;
    }

    public function render_column($column, $post_id)
    {
        if ($column === 'serpforge_links') {
            $int = get_post_meta($post_id, 'serpforge_internal_links', true) ?: 0;
            $ext = get_post_meta($post_id, 'serpforge_external_links', true) ?: 0;
            echo "<span style='color:green' title='Internal'>Int: $int</span> | <span style='color:orange' title='External'>Ext: $ext</span>";
        }
    }
}
