<?php
namespace SerpForge\Meta;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Handles Meta Tags (Title, Description, Robots).
 */
class MetaManager
{

    public function __construct()
    {
        add_filter('pre_get_document_title', [$this, 'filter_document_title'], 15);
        add_action('wp_head', [$this, 'output_meta_tags'], 1);
        add_action('wp_head', [$this, 'output_social_tags'], 2);
    }

    /**
     * Filter the document title.
     * 
     * @param string $title Existing title.
     * @return string Modified title.
     */
    public function filter_document_title($title)
    {
        $sep = get_option('serpforge_separator', '-');

        if (is_front_page()) {
            $home_title = get_option('serpforge_home_title', '');
            if (!empty($home_title)) {
                return $this->replace_variables($home_title, $sep);
            }
        }

        if (is_singular()) {
            global $post;
            $custom_title = get_post_meta($post->ID, '_serpforge_seo_title', true);

            if (!empty($custom_title)) {
                return $this->replace_variables($custom_title, $sep);
            }

            // Fallback to default structure
            $type = get_post_type();
            $default = ($type === 'page')
                ? get_option('serpforge_page_title_default', '%title% %sep% %sitename%')
                : get_option('serpforge_post_title_default', '%title% %sep% %sitename%');

            return $this->replace_variables($default, $sep);
        }

        return $title;
    }

    public function output_meta_tags()
    {
        echo "\n<!-- SERPforge SEO Plugin -->\n";

        // Webmaster Verification
        $google = get_option('serpforge_google_verify');
        $bing = get_option('serpforge_bing_verify');
        if ($google)
            echo '<meta name="google-site-verification" content="' . esc_attr($google) . '" />' . "\n";
        if ($bing)
            echo '<meta name="msvalidate.01" content="' . esc_attr($bing) . '" />' . "\n";

        $desc = '';
        $robots = [];

        if (is_front_page()) {
            $desc = get_option('serpforge_home_desc', '');
        } elseif (is_singular()) {
            global $post;
            $desc = get_post_meta($post->ID, '_serpforge_seo_description', true);
            $robots = get_post_meta($post->ID, '_serpforge_robots', true);
        }

        // Output Description
        if (!empty($desc)) {
            $desc = $this->replace_variables($desc);
            echo '<meta name="description" content="' . esc_attr($desc) . '" />' . "\n";
        }

        // Output Robots
        if (!empty($robots) && is_array($robots)) {
            echo '<meta name="robots" content="' . esc_attr(implode(',', $robots)) . '" />' . "\n";
        }
    }

    public function output_social_tags()
    {
        if (!is_singular()) {
            return;
        }

        global $post;
        $sep = get_option('serpforge_separator', '-');

        // Defaults
        $default_title = $this->replace_variables(get_option('serpforge_post_title_default', '%title% %sep% %sitename%'), $sep);
        $default_desc = get_post_meta($post->ID, '_serpforge_seo_description', true);

        // OpenGraph Data
        $og_title = get_post_meta($post->ID, '_serpforge_og_title', true) ?: $default_title;
        $og_desc = get_post_meta($post->ID, '_serpforge_og_description', true) ?: $default_desc;
        $og_image = get_post_meta($post->ID, '_serpforge_og_image', true);

        // Twitter Data
        $tw_title = get_post_meta($post->ID, '_serpforge_twitter_title', true) ?: $og_title;
        $tw_desc = get_post_meta($post->ID, '_serpforge_twitter_description', true) ?: $og_desc;
        $tw_image = get_post_meta($post->ID, '_serpforge_twitter_image', true) ?: $og_image;

        // Fallback Image (Featured)
        if (empty($og_image) && has_post_thumbnail($post->ID)) {
            $og_image = get_the_post_thumbnail_url($post->ID, 'full');
        }
        if (empty($tw_image)) {
            $tw_image = $og_image;
        }

        echo "\n<!-- SERPforge Social Tags -->\n";

        // OG Output
        echo '<meta property="og:type" content="article" />' . "\n";
        echo '<meta property="og:title" content="' . esc_attr($og_title) . '" />' . "\n";
        echo '<meta property="og:description" content="' . esc_attr($og_desc) . '" />' . "\n";
        echo '<meta property="og:url" content="' . esc_attr(get_permalink($post->ID)) . '" />' . "\n";
        echo '<meta property="og:site_name" content="' . esc_attr(get_bloginfo('name')) . '" />' . "\n";
        if ($og_image)
            echo '<meta property="og:image" content="' . esc_attr($og_image) . '" />' . "\n";

        // Twitter Output
        echo '<meta name="twitter:card" content="summary_large_image" />' . "\n";
        echo '<meta name="twitter:title" content="' . esc_attr($tw_title) . '" />' . "\n";
        echo '<meta name="twitter:description" content="' . esc_attr($tw_desc) . '" />' . "\n";
        if ($tw_image)
            echo '<meta name="twitter:image" content="' . esc_attr($tw_image) . '" />' . "\n";
    }

    private function replace_variables($text, $sep = '-')
    {
        $text = str_replace('%sitename%', get_bloginfo('name'), $text);
        $text = str_replace('%sitedesc%', get_bloginfo('description'), $text);
        $text = str_replace('%title%', get_the_title(), $text);
        $text = str_replace('%sep%', $sep, $text);
        return $text;
    }
}
