<?php
namespace SerpForge\Schema;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Handles JSON-LD Schema markup.
 */
class SchemaManager
{

    public function __construct()
    {
        add_action('wp_head', [$this, 'output_schema'], 20);
    }

    /**
     * Output JSON-LD script.
     */
    public function output_schema()
    {
        $schema = $this->generate_schema();
        if (!empty($schema)) {
            echo '<script type="application/ld+json">' . wp_json_encode($schema) . '</script>';
        }
    }

    /**
     * Generate the schema array based on current context.
     *
     * @return array
     */
    private function generate_schema()
    {
        if (is_single() || is_page()) {
            return $this->get_singular_schema();
        }

        if (is_front_page()) {
            return $this->get_organization_schema();
        }

        return [];
    }

    /**
     * Get Organization/Person Schema (Knowledge Graph).
     */
    private function get_organization_schema()
    {
        $type = get_option('serpforge_schema_entity_type', 'Organization');
        $name = get_option('serpforge_knowledge_name', get_bloginfo('name'));
        $logo = get_option('serpforge_knowledge_logo', get_site_icon_url(512));

        $schema = [
            '@type' => $type,
            'name' => $name,
            'url' => home_url(),
        ];

        if ($logo) {
            $schema['logo'] = [
                '@type' => 'ImageObject',
                'url' => $logo
            ];
            // If Person, use 'image' instead of 'logo' generally, but Schema.org flexible. 
            // For Person, 'image' is better.
            if ($type === 'Person') {
                $schema['image'] = $logo;
                unset($schema['logo']);
            }
        }

        return $schema;
    }

    /**
     * Get Article Schema.
     */
    private function get_singular_schema()
    {
        global $post;

        $type = get_post_meta($post->ID, '_serpforge_schema_type', true) ?: 'Article';
        $headline = get_post_meta($post->ID, '_serpforge_schema_headline', true) ?: get_the_title($post);
        $desc = get_post_meta($post->ID, '_serpforge_schema_description', true) ?: get_the_excerpt($post);

        $schema = [
            '@context' => 'https://schema.org',
            '@type' => $type,
            'headline' => $headline,
            'description' => $desc,
            'datePublished' => get_the_date('c', $post),
            'dateModified' => get_the_modified_date('c', $post),
            'author' => [
                '@type' => 'Person',
                'name' => get_the_author_meta('display_name', $post->post_author),
                'url' => get_author_posts_url($post->post_author),
            ],
            'mainEntityOfPage' => [
                '@type' => 'WebPage',
                '@id' => get_permalink($post),
            ],
            'publisher' => $this->get_organization_schema()
        ];

        // Add Image if available
        if (has_post_thumbnail($post)) {
            $schema['image'] = [
                '@type' => 'ImageObject',
                'url' => get_the_post_thumbnail_url($post, 'full'),
            ];
        }

        return $schema;
    }
}
