<?php
namespace SerpForge\Sitemap;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Handles XML Sitemap generation.
 */
class SitemapGenerator
{

    public function __construct()
    {
        add_action('init', [$this, 'add_rewrite_rules']);
        add_filter('query_vars', [$this, 'add_query_vars']);
        add_action('template_redirect', [$this, 'render_sitemap']);
        add_filter('wp_sitemaps_enabled', '__return_false');
        add_action('init', [$this, 'debug_rewrite_rules'], 999);
    }

    public function add_rewrite_rules()
    {
        add_rewrite_rule('sitemap_index\.xml$', 'index.php?serpforge_sitemap=index', 'top');
        add_rewrite_rule('([a-z0-9_-]+)-sitemap\.xml$', 'index.php?serpforge_sitemap=$matches[1]', 'top');
    }

    public function add_rewrite_rules_filter($rules)
    {
        // Fallback filter if standard init hook fails
        $sitemap_rules = [
            'sitemap_index\.xml$' => 'index.php?serpforge_sitemap=index',
            '([a-z0-9_-]+)-sitemap\.xml$' => 'index.php?serpforge_sitemap=$matches[1]',
        ];
        return $sitemap_rules + $rules;
    }

    public function add_query_vars($vars)
    {
        $vars[] = 'serpforge_sitemap';
        return $vars;
    }


    public function render_sitemap()
    {
        $type = get_query_var('serpforge_sitemap');
        if (!$type) {
            return;
        }

        // Always render if we matched the query var
        header('Content-Type: application/xml; charset=utf-8');
        echo '<?xml version="1.0" encoding="UTF-8"?>';
        echo '<?xml-stylesheet type="text/xsl" href="' . esc_url(SERPFORGE_URL . 'assets/xsl/sitemap.xsl?ver=' . time()) . '"?>';

        if ($type === 'index') {
            $this->render_index();
        } else {
            $this->render_sub_sitemap($type);
        }
        exit;
    }

    private function render_index()
    {
        echo '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';

        $post_types = get_post_types(['public' => true], 'objects');
        foreach ($post_types as $pt) {
            if ($pt->name === 'attachment')
                continue;

            $opt_key = "serpforge_sitemap_include_{$pt->name}";
            $default = in_array($pt->name, ['post', 'page']) ? '1' : '0';
            $enabled = get_option($opt_key, $default);

            if ($enabled === '1') {
                echo '<sitemap>';
                echo '<loc>' . home_url('/' . $pt->name . '-sitemap.xml') . '</loc>';
                echo '<lastmod>' . date('c') . '</lastmod>';
                echo '</sitemap>';
            }
        }

        echo '</sitemapindex>';
    }

    private function render_sub_sitemap($post_type)
    {
        // Validation: Ensure valid public post type
        $pt_obj = get_post_type_object($post_type);
        if (!$pt_obj || !$pt_obj->public) {
            // If invalid sitemap requested, 404
            status_header(404);
            // We already output headers, so just stop XML or output error?
            // Since we output content-type xml, let's just exit or empty set.
            exit;
        }

        echo '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';

        $args = [
            'numberposts' => 1000,
            'post_status' => 'publish',
            'post_type' => $post_type,
            'orderby' => 'modified',
            'order' => 'DESC'
        ];

        // Exclusions
        $excluded = get_option('serpforge_sitemap_exclude_ids', '');
        if (!empty($excluded)) {
            $args['exclude'] = array_map('intval', explode(',', $excluded));
        }

        $posts = get_posts($args);

        if ($post_type === 'page' || $post_type === 'post') {
            // Home URL usually goes in page-sitemap or a dedicated one?
            // Let's put it in the 'page' sitemap if enabled, or just hardcode it here
            if ($post_type === 'page') {
                $this->add_url(home_url('/'), '1.0', 'daily', date('c'));
            }
        }

        foreach ($posts as $post) {
            $permalink = get_permalink($post);
            $modified = get_the_modified_date('c', $post);
            $this->add_url($permalink, '0.8', 'weekly', $modified);
        }

        echo '</urlset>';
    }

    private function add_url($loc, $priority, $freq, $lastmod)
    {
        echo '<url>';
        echo '<loc>' . esc_url($loc) . '</loc>';
        echo '<lastmod>' . esc_html($lastmod) . '</lastmod>';
        echo '<changefreq>' . esc_html($freq) . '</changefreq>';
        echo '<priority>' . esc_html($priority) . '</priority>';
        echo '</url>';
    }

    public function debug_rewrite_rules()
    {
        if (isset($_GET['serpforge_debug_rules'])) {
            global $wp_rewrite;
            echo '<pre>';
            echo "<h3>SERPforge Sitemap Rules Debug</h3>";

            $rules = $wp_rewrite->wp_rewrite_rules();
            $found = false;

            if ($rules) {
                foreach ($rules as $pattern => $query) {
                    if (strpos($pattern, 'sitemap') !== false) {
                        echo "<strong>Found Rule:</strong> " . esc_html($pattern) . " <br/>&nbsp;&nbsp;=> " . esc_html($query) . "<br/>";
                        $found = true;
                    }
                }
            }

            if (!$found) {
                echo "<strong>NO SITEMAP RULES FOUND!</strong><br/>";
                echo "Current rules count: " . count($rules) . "<br/>";
            }

            echo "\n<h3>Registered Public Query Vars</h3>";
            global $wp;
            var_dump($wp->public_query_vars);

            echo '</pre>';
            exit;
        }
    }
}
